// Topic 7: Using polymorphism effectively

// This is a small sample of the Builder design pattern

#include <stdio.h>
#include <string.h>

///////////////////////////////////////////////////////////////////////
class CFileBuilder
{
	public:

		CFileBuilder(char filename[256], FILE *pStream=NULL)
		{
			// If no stream was given, then open a file stream
			if (pStream != NULL)
				m_pStream = pStream;
			else
				m_pStream = fopen(filename, "w");
		}

		virtual ~CFileBuilder()
		{
			fclose(m_pStream); // Close the file stream
		}

		// Virtual functions...
		virtual void Header(char text[])	{  }
		virtual void Body(char text[])		{  }
		virtual void Footer(char text[])	{  }

		// Simple function that outputs text to the stream
		void Output(char text[])
		{ 
			fprintf(m_pStream, text); 
			fflush(m_pStream);
		}

	protected:
		// Data members used by child classes
		FILE* m_pStream;
};

///////////////////////////////////////////////////////////////////////
class CTextFileBuilder : public CFileBuilder
{
	public:
		// Constructor simply calls base class constructor
		CTextFileBuilder(char filename[256], FILE *pStream=NULL) 
			: CFileBuilder(filename, pStream)
		{
		}

		virtual void Header(char text[])
		{
			Output("*** ");
			Output(text);
			Output(" ***\n\n");
		}

		virtual void Body(char text[])
		{
			Output(text);
			Output("\n\n");
		}

		virtual void Footer(char text[])
		{ 
			Output(text);
			Output("\n");
		}

};

///////////////////////////////////////////////////////////////////////
class ConsoleOutputBuilder : public CTextFileBuilder
{
	public:
		// No filename required in this constructor,
		// but we do pass in 'stdout' as the file stream
		ConsoleOutputBuilder() : CTextFileBuilder("", stdout)
		{
		}
};

///////////////////////////////////////////////////////////////////////
class CHtmlBuilder : public CFileBuilder
{
	public:
		// Constructor simply calls base class constructor
		CHtmlBuilder(char filename[256], FILE *pStream=NULL) : CFileBuilder(filename, pStream)
		{
		}

		// Virtual functions...
		virtual void Header(char text[])
		{
			Output("<HTML>\n");
			Output("\t<HEADER>\n");
			Output("\t\t<TITLE>");
			Output(text);
			Output("</TITLE>\n");
			Output("\t</HEADER>\n\n");
		}		
		
		virtual void Body(char text[])
		{
			Output("\t<BODY>\n");
			Output("\t\t");
			Output(text);
			Output("\n\t</BODY>\n");
		}

		virtual void Footer(char text[])
		{	
			Output("</HTML>");
		}
};

///////////////////////////////////////////////////////////////////////
void main()
{
	// Create an array of the 3 different builders
	CFileBuilder* pFileBuilder[3];
	pFileBuilder[0] = new CTextFileBuilder("C:\\textbuild.txt");
	pFileBuilder[1] = new ConsoleOutputBuilder;
	pFileBuilder[2] = new CHtmlBuilder("C:\\htmlbuild.html");

	int i; // Counter for loops

	// Loop through and polymorphically build...
	for (i=0; i<3; ++i)
	{
		pFileBuilder[i]->Header("This is the header!");
		pFileBuilder[i]->Body("This is the body!");
		pFileBuilder[i]->Footer("This is the footer!");
	}

	// Clean up
	for (i=0; i<3; ++i)
		delete pFileBuilder[i];
	
}
